<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Query\SelectQuery;
use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;
use App\Text\Messages;
use ArrayObject;

/**
 * Users Model
 *
 * @property \App\Model\Table\RolesTable&\Cake\ORM\Association\BelongsTo $Roles
 *
 * @method \App\Model\Entity\User newEmptyEntity()
 * @method \App\Model\Entity\User newEntity(array $data, array $options = [])
 * @method array<\App\Model\Entity\User> newEntities(array $data, array $options = [])
 * @method \App\Model\Entity\User get(mixed $primaryKey, array|string $finder = 'all', \Psr\SimpleCache\CacheInterface|string|null $cache = null, \Closure|string|null $cacheKey = null, mixed ...$args)
 * @method \App\Model\Entity\User findOrCreate($search, ?callable $callback = null, array $options = [])
 * @method \App\Model\Entity\User patchEntity(\Cake\Datasource\EntityInterface $entity, array $data, array $options = [])
 * @method array<\App\Model\Entity\User> patchEntities(iterable $entities, array $data, array $options = [])
 * @method \App\Model\Entity\User|false save(\Cake\Datasource\EntityInterface $entity, array $options = [])
 * @method \App\Model\Entity\User saveOrFail(\Cake\Datasource\EntityInterface $entity, array $options = [])
 * @method iterable<\App\Model\Entity\User>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\User>|false saveMany(iterable $entities, array $options = [])
 * @method iterable<\App\Model\Entity\User>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\User> saveManyOrFail(iterable $entities, array $options = [])
 * @method iterable<\App\Model\Entity\User>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\User>|false deleteMany(iterable $entities, array $options = [])
 * @method iterable<\App\Model\Entity\User>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\User> deleteManyOrFail(iterable $entities, array $options = [])
 */
class UsersTable extends Table
{
    /**
     * Initialize method
     *
     * @param array<string, mixed> $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config): void
    {
        parent::initialize($config);

        $this->setTable('users');
        $this->setDisplayField('name');
        $this->setPrimaryKey('id');
        $this->addBehavior('Register');

        $this->belongsTo('Roles', [
            'foreignKey' => 'role_id',
            'joinType' => 'INNER',
        ]);
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator): Validator
    {
        $messageEmpty = Messages::get('REQUIRED');
        $messageEmail = Messages::get('INVALID_EMAIL');
        $messageInteger = Messages::get('MUST_BE_INTEGER');
        $messageRangePassword = Messages::get('PASSWORD_LENGTH');
        $messageUserUnique = Messages::get('USERNAME_EXISTS');
        $messageEmailUnique = Messages::get('EMAIL_EXISTS');
        $messageRutNotValid = Messages::get('INVALID_RUT');
        $messageRutUnique = Messages::get('RUT_EXISTS');


        $validator
            ->scalar('name')
            ->maxLength('name', 255)
            ->requirePresence('name', 'create', $messageEmpty)
            ->notEmptyString('name', $messageEmpty);

        $validator
            ->scalar('last_name')
            ->maxLength('last_name', 255)
            ->requirePresence('last_name', 'create', $messageEmpty)
            ->notEmptyString('last_name', $messageEmpty);

        $validator
            ->requirePresence('email','create',$messageEmpty)
            ->notEmptyString('email',$messageEmpty)
            ->add('email','validFormat',[
                'rule' => 'email',
                'message' => $messageEmail,
            ])
            ->add('email', 'unique', ['rule' => 'validateUnique', 'provider' => 'table','message' => $messageEmailUnique]);

        $validator
            ->scalar('password')
            ->maxLength('password', 20, $messageRangePassword)
            ->minLength('password', 4, $messageRangePassword)
            ->allowEmptyString('password', null, function ($context) {
                // Permitir vacío en edición, pero requerir en creación
                return $context['newRecord'] === false;
            })
            ->requirePresence('password', 'create', $messageEmpty)
            ->notEmptyString('password', $messageEmpty, function ($context) {
                // Solo requerir si es creación
                return $context['newRecord'] === true;
            });


        $validator
            ->scalar('rut')
            ->requirePresence('rut', 'create',$messageEmpty)
            ->notEmptyString('rut',$messageEmpty)
            ->add('rut', 'unique', ['rule' => 'validateUnique', 'provider' => 'table','message' => $messageRutUnique]);

        $validator->add('rut', 'custom', [
            'rule' => 'validateRutForm',
            'provider' => 'table',
            'message' =>  $messageRutNotValid
        ]);

        $validator
            ->integer('role_id', $messageInteger)
            ->notEmptyString('role_id', $messageEmpty);


        return $validator;
    }

    /**
     * Returns a rules checker object that will be used for validating
     * application integrity.
     *
     * @param \Cake\ORM\RulesChecker $rules The rules object to be modified.
     * @return \Cake\ORM\RulesChecker
     */
    public function buildRules(RulesChecker $rules): RulesChecker
    {
        $messageNotFound = Messages::get('VALUE_NOT_FOUND');
        $rules->add($rules->isUnique(['email'], ['allowMultipleNulls' => true]), ['errorField' => 'email']);
        $rules->add($rules->isUnique(['rut'], ['allowMultipleNulls' => true]), ['errorField' => 'rut']);
        $rules->add($rules->existsIn(['role_id'], 'Roles'), ['errorField' => 'role_id', 'message' => $messageNotFound]);

        return $rules;
    }
    public function findForAuthentication(\Cake\ORM\Query $query, array $options): \Cake\ORM\Query
    {
        return $query
                ->where(
                    [
                        'active' => 1
                    ]
                )
                ->contain(
                    [
                        'Roles'
                    ]
                );
    }
    public function beforeMarshal(\Cake\Event\EventInterface $event, ArrayObject $data, ArrayObject $options): void
    {
        if (!empty($data['rut']))
        {
            $data['rut'] = str_replace(['.', ' '], '', $data['rut']);
        }
    }
    /**
     * Validar Número de documento chileno
     *
     */
    public function validateRut($rut)
    {
        $rut = strval($rut);
        $arrayRut = explode('-', $rut);
        if(!isset($arrayRut[0]) || !isset($arrayRut[1]))
        {
            return null;
        }
        $arrayRut[0] = str_replace('.', '', $arrayRut[0]);
        $arrayRut[0] = preg_replace('/[\x00-\x1F\x7F\xA0\x{200B}\s]+/u', ' ', $arrayRut[0]);
        $arrayRut[1] = preg_replace('/[\x00-\x1F\x7F\xA0\x{200B}\s]+/u', ' ', $arrayRut[1]);
        $rut = trim($arrayRut[0]).'-'.trim($arrayRut[1]);
        if (!preg_match("/^[0-9.]+[-]?+[0-9kK]{1}/", $rut))
        {
            return null;
        }
        $rut = preg_replace('/[\.\-]/i', '', $rut);
        $dv = substr($rut, -1);
        $numero = substr($rut, 0, strlen($rut) - 1);
        $i = 2;
        $suma = 0;
        foreach (array_reverse(str_split($numero)) as $v) {
            if ($i == 8)
                $i = 2;
            $suma += $v * $i;
            ++$i;
        }
        $dvr = 11 - ($suma % 11);

        if ($dvr == 11)
            $dvr = 0;
        if ($dvr == 10)
            $dvr = 'K';
        if ($dvr == strtoupper($dv))
            return $rut;
        else
            return null;
    }
    public function validateRutForm($rut)
    {
        if($this->validateRut($rut))
        {
            return true;
        }
        return false;
    }
}
